;;; -*- Mode: Lisp; Package: EDITOR -*-
;;; powershell-mode.l --- Mode for editing Powershell 2.0 scripts

;; Copyright (C) 2011 kazy <kazy@kazy111.info>
;; Copyright (C) 2009, 2010 Frederic Perrin

;; Author: kazy <kazy@kazy111.info>
;; Author(base script): Frederic Perrin <frederic (dot) perrin (arobas) resel (dot) fr>
;; Keywords: Powershell, Monad, MSH
;; Version: 0.2.2

;; Indentation algorithm is based on Emacs PowerShell-mode.
;; Original version is here: http://www.emacswiki.org/emacs/PowerShell

;; This file is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation, either version 3 of the License,
;; or (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <http://www.gnu.org/licenses/>.


;; History
;; v 0.2.2  <2011/12/17 16:04:30 +0900>
;;  fix freeze (first line continuation)
;;  implement list-function
;; v 0.2.1  <2011/12/14 19:31:59 +0900
;;  improve completion (complete pointed word)
;;  indent-region now working
;;  implement selected text indent (select text by drag, and push TAB)
;; v 0.2    <2011/12/08 20:05:34 +0900>
;;  support powershell execution
;;    * file or selected text (C-M-x)
;;    * region text           (C-M-z)
;;  ignore case in completion
;; v 0.1.1  <2011/12/07 16:52:12 +0900>
;;  fix multi-comment, symbol syntax
;; v 0.1    <2011/12/06 16:08:23 +0900>
;;  first release

(provide "powershell-mode")

(in-package "editor")

(export '(powershell-mode
		  powershell-indent-line
		  *powershell-keyword-file*
		  *powershell-edit-mode-hook*
		  *powershell-edit-mode-syntax-table*
		  *powershell-edit-mode-map*
		  *powershell-edit-mode-abbrev-table*))

(defvar powershell-indent 4
  "Amount of horizontal space to indent after, for instance, an opening brace")

(defvar powershell-comment-indent 4)

(defvar *powershell-indent-tabs-mode* t
  "Using tab in indent")

(defvar powershell-continuation-indent 2
  "Amount of horizontal space to indent a continuation line")

(defvar powershell-continued-regexp  ".*\\(|[\\t ]*\\|`\\)$"
  "Regexp matching a continued line (ending either with an explicit backtick, or with a pipe)."
  )

; Execute command
(defvar *powershell-cmd* "powershell.exe")
(defvar *powershell-execution-message* "Running PowerShell, please wait...")

; Keyword filename
(defvar *powershell-keyword-file* "PowerShell")

; Comment color
(defvar *powershell-comment-color* '(:keyword :comment))
; Variable color
(defvar *powershell-variable-color* '(:keyword 2))



(defvar *powershell-edit-mode-abbrev-table* nil)
(defvar *powershell-keyword-hash-table* nil)
(defvar *powershell-regexp-keyword-list*
  nil
  "powershellt-mode: RexExp keyword list")
(defvar *powershell-completion-list* nil)

(setq *powershell-regexp-keyword-list*
	  (compile-regexp-keyword-list
	   `(
		 ; comment
		 ( "#.*" t ,*powershell-comment-color*)
		 ; variable
		 ("\$[a-zA-Z0-9_\.\:\-]*" nil ,*powershell-variable-color*)

		 )))

; For emacs compatibillity
(defun current-indentation ()
  (save-excursion
	(beginning-of-line)
	(skip-chars-forward " \t")
	(current-column)))

(defun indent-line-to (column)
  (smart-indentation column))

; ------------------------------
(defun powershell-newline-and-indent (&optional (arg 1))
  "powershell-mode: newline + indent"
  (interactive "*p")
  (delete-trailing-spaces)
  (insert #\LFD arg)
  (powershell-indent-line))

; ------------------------------

(defun powershell-electric-close (&optional (arg 1))
  (interactive "*p")
  (self-insert-command arg)
  (powershell-indent-line)
  (save-excursion
	(forward-char -1)
	(and (goto-matched-parenthesis)
		 (show-matched-parenthesis)))
  t)


(defun powershell-continuation-line-p ()
  "Returns t is the current line is a continuation line (i.e. the
previous line is a continued line, ending with a backtick or a pipe"
  (interactive)
  (save-excursion
	(forward-line -1)
	(while (and (powershell-blank-line-p)
				(not (bobp)))
	  (forward-line -1))
	(looking-at powershell-continued-regexp)))

(defun powershell-blank-line-p ()
  "Returns t is the current line is a blank line"
  (interactive)
  (save-excursion
	(beginning-of-line)
	(skip-chars-forward " \t")
	(eolp)))

(defun powershell-indent-line-amount ()
  "Returns the column to which the current line ought to be indented."
  (interactive)
  (save-excursion
	(beginning-of-line)
	(let ((closing-paren (looking-at "[\t ]*[])}]")))
	  ;; a very simple indentation method: if on a continuation line (i.e. the
	  ;; previous line ends with a trailing backtick or pipe), we indent relative
	  ;; to the continued line; otherwise, we indent relative to the ([{ that
	  ;; opened the current block.
	  (if (powershell-continuation-line-p)
		  (progn
			(while (and (powershell-continuation-line-p)
						(not (bobp)))
			  (forward-line -1))
			(+ (current-indentation) powershell-continuation-indent))
		;(condition-case nil
		(handler-case
			(progn
			  (if (backward-up-list 1 t)
				  ;; indentation relative to the opening paren: if there is text (no
				  ;; comment) after the opening paren, vertically align the block
				  ;; with this text; if we were looking at the closing paren, reset
				  ;; the indentation; otherwise, indent the block by powershell-indent.
				  (cond ((not (looking-at ".[\t ]*\\(#.*\\)?$"))
						 (forward-char)
						 (skip-chars-forward " \t")
						 (current-column))
						(closing-paren
						 (current-indentation))
						(t
						 (+ (current-indentation) powershell-indent)))
				0))
		  )))))


(defun powershell-indent-line-gui ()
  (interactive)

  (selection-start-end (start end)
	(save-excursion
	  (indent-region start end)))
  (powershell-indent-line)
  (powershell-completion))

(defun powershell-indent-line ()
  "Indent the current line of powershell mode, leaving the point
in place if it is inside the meat of the line"
  (interactive)
  
  (case (save-excursion
		  (goto-bol)
		  (parse-point-syntax))
	(:string)
	(:comment
	 (let ((column (calc-powershell-comment-indent)))
	   (when (integerp column)
		 (smart-indentation column))))
	(t
	 (let ((savep (> (current-column) (current-indentation)))
		   (amount (powershell-indent-line-amount)))
	   (if savep
		   (save-excursion (indent-line-to amount))
		   (indent-line-to amount))))))


(defvar-local powershell-comment-indent-variable 'powershell-comment-indent)

(defun calc-powershell-comment-indent ()
  "Calculate indent size when current line in multi-comment"
  (save-excursion
	(goto-bol)
	(skip-chars-forward " \t")
	(let ((eolp (eolp)))
	  (when (and (or eolp (looking-for "#"))
				 (scan-buffer "<#" :reverse t))
		(while (and (eq (parse-point-syntax) ':comment)
					(scan-buffer "<#" :reverse t :no-dup t)))
		(+ (current-column)
		   (if eolp
			   (if (symbolp powershell-comment-indent-variable)
				   (symbol-value powershell-comment-indent-variable)
				   0)
			   1))))))

; syntax table
(defvar *powershell-edit-mode-syntax-table* nil)
(unless *powershell-edit-mode-syntax-table*
  (setq *powershell-edit-mode-syntax-table* (make-syntax-table))

  (dotimes (x 127)
	(let ((c (code-char x)))
	  (unless (or (alphanumericp c)
				  (eq #\[ c)
				  (eq #\] c))
		(set-syntax-symbol *powershell-edit-mode-syntax-table* c))))
  (set-syntax-symbol *powershell-edit-mode-syntax-table* #\:)
  (set-syntax-symbol *powershell-edit-mode-syntax-table* #\_)
  (set-syntax-symbol-prefix *powershell-edit-mode-syntax-table* #\$)
  (set-syntax-string *powershell-edit-mode-syntax-table* #\")
  (set-syntax-string *powershell-edit-mode-syntax-table* #\')
  (set-syntax-whitespace *powershell-edit-mode-syntax-table* #\SPC)
  (set-syntax-whitespace *powershell-edit-mode-syntax-table* #\TAB)
  (set-syntax-whitespace *powershell-edit-mode-syntax-table* #\RET)
  (set-syntax-whitespace *powershell-edit-mode-syntax-table* #\C-l)
  (set-syntax-start-comment *powershell-edit-mode-syntax-table* #\# nil)
  (set-syntax-end-comment *powershell-edit-mode-syntax-table* #\LFD nil t)
  (set-syntax-start-multi-comment *powershell-edit-mode-syntax-table* "<#")
  (set-syntax-end-multi-comment *powershell-edit-mode-syntax-table* "#>")


  (set-syntax-match *powershell-edit-mode-syntax-table* #\( #\))
  (set-syntax-match *powershell-edit-mode-syntax-table* #\[ #\])
  (set-syntax-match *powershell-edit-mode-syntax-table* #\{ #\})
  (set-syntax-escape *powershell-edit-mode-syntax-table* #\`))



; completion
(defun powershell-completion ()
  (interactive)
  (or (setq *powershell-completion-list*
			(make-list-from-keyword-table *powershell-keyword-hash-table*))
	  (return-from powershell-completion nil))
  (when (skip-syntax-spec-backward "w_")
	(let ((from (point)))
	  (skip-syntax-spec-forward "w_")
	  (do-completion from (point) :list-ignore-case *powershell-completion-list* nil nil 'always))))

; execute current file
(defun powershell-execution-normal ()
  "execute file or selected text"
  (interactive)
  (let ((fmtstr "~A ~A")
		(cmd (get-buffer-file-name)))
	(selection-start-end (start end)
	  (setq cmd (buffer-substring start end))
	  (setq fmtstr "~A -command \"~A\""))
	(powershell-execution fmtstr cmd)))

(defun powershell-execution-region ()
  "execute region"
  (interactive)
  (let ((fmtstr "~A ~A")
		(cmd (buffer-substring (region-beginning) (region-end))))
	(powershell-execution fmtstr cmd)))

(defun powershell-execution (fmtstr cmd)
  (interactive)
  (minibuffer-message "~A" *powershell-execution-message*)
  (let ((outfile nil)
		(buffer nil))
	(unwind-protect
		(save-excursion
		  (setq outfile (make-temp-file-name))
		  (call-process (format nil fmtstr *powershell-cmd* cmd)
						:output outfile :show :hide :wait t)
		  (setq buffer (get-buffer-create "*PS-Out*"))
		  (set-buffer buffer)
		  (setup-temp-buffer buffer)
		  (goto-char (point-max))
		  (if (not (bobp))
			  (insert "\n"))
		  (insert (reduce (lambda (x y) (concat x y))
						  (map 'list
							   (lambda (x) (concat "PS> " x "\n"))
							   (split-string cmd "\n"))))
		  (insert-file-contents outfile)
		  )
	  (and outfile
		   (delete-file outfile))
	  (progn
		(if (= (count-windows nil) 1)
			(split-window 20 nil))
		(switch-to-buffer-other-window buffer)
		(goto-char (point-max))
		(other-window)
		(clear-minibuffer-message)
		))))


; key map
(defvar *powershell-edit-mode-map* nil)
(unless *powershell-edit-mode-map*
  (setq *powershell-edit-mode-map* (make-sparse-keymap))
  (define-key *powershell-edit-mode-map* #\TAB 'powershell-indent-line-gui)
  (define-key *powershell-edit-mode-map* #\RET 'powershell-newline-and-indent)
  (define-key *powershell-edit-mode-map* #\C-A 'powershell-completion)
  (define-key *powershell-edit-mode-map* #\C-M-x 'powershell-execution-normal)
  (define-key *powershell-edit-mode-map* #\C-M-z 'powershell-execution-region)
  (define-key *powershell-edit-mode-map* #\) 'powershell-electric-close)
  (define-key *powershell-edit-mode-map* #\] 'powershell-electric-close)
  (define-key *powershell-edit-mode-map* #\} 'powershell-electric-close))


; build function summary
(defvar *powershell-beginning-of-defun-regexp*
  (compile-regexp "^[ \t]*\\(Function\\|Filter\\|Set-Alias\\)[ \t]*\\([A-Za-z0-9_]+\\)" t))

(defun powershell-build-summary-of-functions ()
  (let ((result nil))
    (save-excursion
      (goto-char (point-min))
      (while (scan-buffer *powershell-beginning-of-defun-regexp* :regexp t :case-fold t :tail t)
       (push (list (current-line-number) (match-string 2)) result)))
    (nreverse result)))


(defun powershell-mode ()
  "Major mode for editing PowerShell files"
  (interactive)
  (kill-all-local-variables)
  (setq buffer-mode 'powershell-mode)
  (setq mode-name "PowerShell")
  (make-local-variable 'mode-specific-indent-command)
  (setq mode-specific-indent-command #'powershell-indent-line)
  (use-keymap *powershell-edit-mode-map*)
  (use-syntax-table *powershell-edit-mode-syntax-table*)


  (make-local-variable 'paragraph-start)
  (setq paragraph-start "^$\\|\f")
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'indent-tabs-mode)
  (setq indent-tabs-mode *powershell-indent-tabs-mode*)
  ; TODO implement tag jump
;  (make-local-variable 'tags-find-target)
;  (setq tags-find-target #'powershell-tags-find-target)
;  (make-local-variable 'tags-find-point)
;  (setq tags-find-point #'powershell-tags-find-point)
  (make-local-variable 'build-summary-function)
  (setq build-summary-function 'powershell-build-summary-of-functions)
  (setq *local-abbrev-table* *powershell-edit-mode-abbrev-table*)
  (setq comment-start "# ")
  (setq comment-end "")
  (setq comment-start-skip "#+[ \t]*")
  (setq comment-indent-function 'powershell-comment-indent)
  
  ; keyword
  (make-local-variable 'regexp-keyword-list)
  (setq regexp-keyword-list *powershell-regexp-keyword-list*)
  (and *powershell-keyword-file*
	   (null *powershell-keyword-hash-table*)
	   (setq *powershell-keyword-hash-table*
			 (load-keyword-file *powershell-keyword-file* t)))
  (when *powershell-keyword-hash-table*
	(make-local-variable 'keyword-hash-table)
	(setq keyword-hash-table *powershell-keyword-hash-table*))
  (run-hooks '*powershell-edit-mode-hook*))


